// RATNET2Plugin.cpp starts
// backprop rat brain for spatial symmetry testing
// CGP; 6/24/00
/* Changes
	CGP; 11/12/00; Change to bring up to current rat API. Also changed from
		8 input/output units in place-place model to 9 input units and 9 output
		units to correspond to number of cues in a view from the environment.
		Changed to use database also. This is for evaluation purposes and not directly
		part of model.
	CGP; 12/24/00; Change to use version 2 of rat API.
	CGP; 1/8/01; Changes to use non-messaging version of API.
	CGP; 1/10/01
		I just spent several hours tracking down a bug. The symptom was that sometimes
		the rat would enter the start arm and fail to find the food, and repeatedly
		run back and forth between the center of the maze and the food, failing to find the
		food. The experimenter would not open doors to proceed with the rest of the trial
		because the rat had not yet consumed the food down the start arm, so this would
		produce an infinite loop. The problem was that the experimenter was taking the rat
		off the maze, at the end of the trial, after the experimenter had received the
		last event from the rat (i.e., that the rat went to the center of the maze, or the rat
		ate the food on the arm). In the time interval between the experimenter receiving the event
		and the experimenter taking the rat off the maze, the rat had time to proceed with his
		programmed behavior. The rat looked around, if he was at the center of the maze,
		for an open arm, and (for example, dependent on the timing of the experimenter) then
		blocked when he was taken off the maze. When he went back on the maze, he would expect to have
		the same door open in front of him as when he went off, so he would not check around him, he'd
		just move. The rat thought he was in one stage of the trial, while the experimenter was intending
		another phase of the trial.
		Change: Program the rat to be robust to changes in the doors that are currently open. If the
		rat fails on a move, then the rat will respond appropriately.
*/

// This parameter should be in plugin settings -- when we get the GUI
// plugin settings implemented.
const double LEARNING_RATE = 0.45; // as per Berkeley, Prince, and Gunay (1999)

const int NUMBER_INPUT_UNITS = 9; // same as LANDMARK_VIEW_WIDTH from EnvironmentConstants.h
const int NUMBER_OUTPUT_UNITS = 9;

const int MAX_GET_START_ARM_ERRORS = 20; // max number of "errors" trying to eat start arm food
const int MAX_MOVE_TO_CENTER_ERRORS = 10; // max number of "errors" trying to move to center

#include <math.h>     	// for fabs
#include <iostream.h>
#include <stdlib.h> 	// for srand & random
#include <OS.h> 		// for system_time
#include <SupportDefs.h> 	// for min_c
#include <Debug.h>

#include "Globals.h"
#include "ANN.h"	// for CObXPlex objects
#include "RATNET2Plugin.h"
#include "EnvironmentConstants.h"
#include "DatabaseFieldNames.h"
#include "View.h"
#include "RatStdLib.h"

// needed by API
RatPlugin *instantiate_rat(PortMessage *ratPort, DatabaseMessage *dbMsg, RatEnvMessage *ratEnvMsg,
	DebugServer *bugServer) {
   return new RATNET2RatPlugin(ratPort, dbMsg, ratEnvMsg, bugServer);
}

// Initialize our back prop model.
// The neural net is a place place model with 8 inputs & 8 outputs.
// See Berkeley, Prince, and Gunay (1999).
// Place place model has no hidden layer.
RATNET2RatPlugin::RATNET2RatPlugin(PortMessage *expRatPort, DatabaseMessage *dbMsg,
	RatEnvMessage *ratEnvMsg, DebugServer *bugServer)
	: RatPlugin(expRatPort, dbMsg, ratEnvMsg, bugServer),
		neuralNet(NUMBER_INPUT_UNITS, NUMBER_OUTPUT_UNITS)
{
   initRand();
}

RATNET2RatPlugin::~RATNET2RatPlugin()
{
}

// Rotate the rat until we are facing a door that is open.
// Returns false if no door was open.
bool RATNET2RatPlugin::RotateUntilOpen()
{
	Debug('n', "RATNET2RatPlugin::RotateUntilOpen");

	for (int i=0; i <= 360; i += MIN_ROTATE_ANGLE) {
	  if (RAT_IsOpen()) return true;
	  RAT_Rotate(MIN_ROTATE_ANGLE);
	}
	
	return false;
}


// Get the views from the two available arms. Assumes the view1 & view2 vectors
// have been previously allocated. Start from current rotation, and rotate until
// two different views have been found of arms with open doors.
void RATNET2RatPlugin::GetAlternativeViews(float *view1, float *view2)
{
	int angle;
	
	float *newView;
	
	for (int i=0; i < LANDMARK_VIEW_WIDTH; i++) {
		view1[i] = view2[i] = 0.0;
	}
	
	for (angle=0; angle <= 360; angle += MIN_ROTATE_ANGLE) {
		if (RAT_IsOpen()) break;
		RAT_Rotate(MIN_ROTATE_ANGLE);
	}
	
	if (! RAT_IsOpen()) {
		Debug('e', "RATNET2RatPlugin::GetAlternativeViews: Could not find 1 view");
		return;
	}
	
	// Facing an open door.
	// Get the view.
	newView = RAT_Look();
	CopyView(view1, newView);
	delete newView;
	
	// Now, try to find a second view, different from the first.
	// What we need to do first is to rotate past this current open door so we don't
	// end up getting a slightly different view down the same arm!
	
	// First, slowly rotate until we are past this currently open door.
	for (; angle <= 360; angle++) {
		RAT_Rotate(1);
		if (! RAT_IsOpen()) {
			break; // have rotated past the door that was open
		}
	}

	int differentView = 0;

	// Start from 10 degrees plus the angle we left off at last time.
	for (angle += MIN_ROTATE_ANGLE; angle <= 360; angle += MIN_ROTATE_ANGLE) {
		RAT_Rotate(MIN_ROTATE_ANGLE);
		if (RAT_IsOpen()) {
			// Have an open door. Check if this view is different from view1.
			newView = RAT_Look();
			differentView = (! ViewsEqual(view1, newView));
			if (differentView) {
				break;
			} else {
				delete newView;
			}
		}
	}
	
	if (! differentView) {
		Debug('e', "RATNET2RatPlugin::GetAlternativeViews: Could not find 2nd view");
		return;
	}

	CopyView(view2, newView);
	delete newView;

	// save these in database for later analysis
	SaveViewRecord(view1, 2); // subType 2 for view1
	SaveViewRecord(view2, 3); // subType 3 for view2
	
	DebugView(view1);
	DebugView(view2);
}

// Subtract two vectors and then compute the vector length of the result.
// Computes || v1 - v2 ||.
double RATNET2RatPlugin::VectorDistance(CObXPlex* v1, CObXPlex* v2)
{
  int length = v1->length();
  double sum = 0.0;
  
  ASSERT(length == v2->length());
    
  for (int i=0; i < length; i++) {
    NeuronActivity* act1  = (NeuronActivity*)((*v1)[i]);
    NeuronActivity* act2  = (NeuronActivity*)((*v2)[i]);
    sum += pow(act1->activity - act2->activity, 2);
  }
  
  return pow(sum, 0.5);
}

// Appear to need to give ANN values between -1.0 and +1.0
#define		SCALE_FACTOR		LANDMARK_SCALE_FACTOR

// Scale factor because we don't have float representation in database yet
#define		SCALE_FACTOR_NO_FLOAT		100.0

// Convert a CObXPlex object to a view of the same length
void ConvertToView(float *resultView, CObXPlex* theResult)
{
  ASSERT(LANDMARK_VIEW_WIDTH == theResult->length());
    
  for (int i=0; i < LANDMARK_VIEW_WIDTH; i++) {
  	NeuronActivity* act1  = (NeuronActivity*)((*theResult)[i]);
  	// Including SCALE_FACTOR_NO_FLOAT only because this view is just
  	// being used for output, not for internal processing here...
    resultView[i] = SCALE_FACTOR_NO_FLOAT * SCALE_FACTOR * act1->activity;
  }
}

// Takes a view vector and converts it to a CObXPlex vector
// Caller owns object.
CObXPlex *RATNET2RatPlugin::ConvertFromView(float *view)
{
  CObXPlex* sample;
  
  sample = new CObXPlex();
  
  int numberOfUnits = NUMBER_INPUT_UNITS;
    
  for (int i=0; i < numberOfUnits; i++) {
    sample->add_high(new NeuronActivity((double) view[i]/SCALE_FACTOR));
  }
  
  return sample;
}

void RATNET2RatPlugin::SaveViewRecord(float *view, int16 subType)
{
	BMessage record;
	int i;
	
	i=0;
	record.AddInt16(DBFLD_RECORD_TYPE, 3);
	record.AddInt16(DBFLD_RECORD_SUB_TYPE, subType);
	record.AddInt16(DBFLD_TRIAL_TYPE, trialType);
	record.AddInt16(DBFLD_TRIAL_NUMBER, trialNumber);
	record.AddInt16(DBFLD_RAT_VIEW0, view[i++]);
	record.AddInt16(DBFLD_RAT_VIEW1, view[i++]);
	record.AddInt16(DBFLD_RAT_VIEW2, view[i++]);
	record.AddInt16(DBFLD_RAT_VIEW3, view[i++]);
	record.AddInt16(DBFLD_RAT_VIEW4, view[i++]);
	record.AddInt16(DBFLD_RAT_VIEW5, view[i++]);
	record.AddInt16(DBFLD_RAT_VIEW6, view[i++]);
	record.AddInt16(DBFLD_RAT_VIEW7, view[i++]);
	record.AddInt16(DBFLD_RAT_VIEW8, view[i++]);
	RAT_AddDatabaseRecord(&record);
}

void RATNET2RatPlugin::SaveDistanceRecord(float distA1, float distA2)
{
	BMessage record;

	record.AddInt16(DBFLD_RECORD_TYPE, 3);
	record.AddInt16(DBFLD_RECORD_SUB_TYPE, 4);
	record.AddInt16(DBFLD_TRIAL_TYPE, trialType);
	record.AddInt16(DBFLD_TRIAL_NUMBER, trialNumber);
	record.AddInt16(DBFLD_DIST_ALT1, (int16) (distA1*100.0));
	record.AddInt16(DBFLD_DIST_ALT2, (int16) (distA2*100.0));
	RAT_AddDatabaseRecord(&record);	
}

/* This function gets the view from the end of the start arm. It tries to take care
 * of problems that happen in trying to get to the start arm. E.g., the rat
 * might be taken off the maze partway through trying to make it to the food
 * at the end of the start arm. Eats the food on the start arm.
 * If the rat fails in its attempts too many times, it will just return it's current
 * view. Program code has been structured loosely. Even if it fails one one step,
 * it tries next step. The main goal in this function is to eat a peice of food
 * and return the view from that location. But, usually, in order to find the food
 * this should involve rotating to find the open arm, moving, then eating.
 */
float *RATNET2RatPlugin::GetStartArmView()
{
	int numberErrors = 0;
	bool done = false;
	
  	while (! done) {
		if (! RotateUntilOpen()) {
			Debug('t', "RATNET2RatPlugin::GetStartArmView: Could not find open door");
			numberErrors++;
  		}
  	
  		if (! RAT_Move()) {
  			Debug('t', "RATNET2RatPlugin::GetStartArmView: Could not Move");
  			numberErrors++;
  		}
 
		if (! RAT_FoodHere()) {
  			Debug('t', "RATNET2RatPlugin::GetStartArmView: Could not find food");
  			numberErrors++;
  		}

		if (RAT_EatFood()) {
			done = true; // successfully moved & found food
		} else {
			Debug('t', "RATNET2RatPlugin::GetStartArmView: Failed to eat food");
			numberErrors++;
		}

  		if (numberErrors > MAX_GET_START_ARM_ERRORS) {
  			done = true;
  			// Give up, just return current view
  		}
  	}

	// Look in current direction, get view vector
	return RAT_Look();
}

/* Attempt to move to the center of the maze. If we consumed the food on the start arm
 * and are located on the start arm prior to calling this function, we should be fine.
 */
void RATNET2RatPlugin::MoveToMazeCenter()
{
	int numberErrors = 0;
	bool done = false;
	
	while (! done) {
		if (! RotateUntilOpen()) {
			Debug('t', "RATNET2RatPlugin::MoveToMazeCenter: Could not find open door");
			numberErrors++;
		}
		
		if (RAT_Move()) {
			done = true;
		} else {
			Debug('t', "RATNET2RatPlugin::MoveToMazeCenter: Could not move");
			numberErrors++;
		}
		
		if (numberErrors > MAX_MOVE_TO_CENTER_ERRORS) {
			done = true;
		}
  	}
}

// Training trial for place-place model
void RATNET2RatPlugin::Step()
{
	float *view =  GetStartArmView();
	
	// save this in database for later analysis
	SaveViewRecord(view, 1); // subType 1 for start arm view
	
	// Convert view to neural net input representation
	CObXPlex* startLoc; // vector representation of start location
	startLoc = ConvertFromView(view); // scales by 1/SCALE_FACTOR
	delete view;

  	// Use the current neural net to generate a choice. We'll make this choice
  	// in the environment, then if we are correct (found the food), we'll use
  	// this choice as the correct choice in the backprop training. If we're incorrect
  	// then we'll use the other door that is available as the correct choice in the
  	// back prop training. Since only two doors are available as choices on each trial
  	// we can easily infer the actual correct choice when a mistake is made.
  	// NOTE: This is a major assumption of this model that the correct choice is
  	// inferred after making an error.
 
  	// See what arm choice the current network weights predict with this starting arm.
  	CObXPlex* theSample = startLoc;
  	CObXPlex* theResult = neuralNet.Evaluate(theSample);	

	float resultView[LANDMARK_VIEW_WIDTH]; // temporary, for database output
	ConvertToView(resultView, theResult);
	SaveViewRecord(resultView, 0); // subType 0 for predicted view
	
  	// Now, we need to figure out which of the available maze arms is closest
  	// to our chosen arm. This will be the one that we pick. Let A1 and A2
  	// be the two available arms, and let C be our chosen arm, each
  	// represented as vectors. We'll compute
  	// || A1 - C|| and ||A2 - C||. We'll chose the arm that has the smallest
  	// diference form C. (In case of equality, we'll chose randomly.)
  
  	// Get the two available arms in vector format. Because we have chosen
  	// the (sample) arm already, the maze now has the two alternative arms open.
  	//rat_direction arm1, arm2;
  	
  	CObXPlex *A1, *A2;
  	float view1[LANDMARK_VIEW_WIDTH], view2[LANDMARK_VIEW_WIDTH];
  	
  	MoveToMazeCenter();
  	
	GetAlternativeViews(view1, view2);
	
	A1 = ConvertFromView(view1); // scales by 1/SCALE_FACTOR
	A2 = ConvertFromView(view2); // scales by 1/SCALE_FACTOR
/*  
  DOOR angleDoor1 = BrainUtils::convertFromDirectionToAngle(arm1);
  A1 = BrainUtils::convertFromAngleToVector(angleDoor1, UNIT_ANGLE);
  DOOR angleDoor2 = BrainUtils::convertFromDirectionToAngle(arm2);
  A2 = BrainUtils::convertFromAngleToVector(angleDoor2, UNIT_ANGLE);
*/

  	// Compute distances of start arm view (C) from the alternative arm views (A1, A2).
  	double distA1 = VectorDistance(A1, theResult);
  	double distA2 = VectorDistance(A2, theResult);
  	
  	SaveDistanceRecord(distA1, distA2);
  
  	CObXPlex *choice, *notChosen;
  
  	if (distA1 == distA2) {
  		// Not likely, but just in case the C result vector is
  		// equidistant between A1 and A2, choose from the alternatives at random.
  		if (random()%2) {
      		choice = A2;
      		notChosen = A1;
   		} else {
   	  		choice = A1;
      		notChosen = A2;
  		}
  	} else if (distA1 > distA2) {
    	// greater distance from A1, so choose A2
    	choice = A2;
    	notChosen = A1;
  	} else { // greater distance from A2, so choose A1
    	choice = A1;
    	notChosen = A2;
  	}

	// Now, turn the rat so he's heading towards the view that we have chosen
	if (choice == A1) {
		RAT_RotateUntilView(view1);
	} else {
		RAT_RotateUntilView(view2);
	}

	// Enter that arm
	if (RAT_IsOpen()) {
  		RAT_Move();
  	} else {
  		Debug('e', "RATNET2RatPlugin::Step: ERROR: Selected arm choice is not open!");
  	}
  
  	// find out if the choice was correct
  	CObXPlex *correctChoice;
  
  	if (RAT_FoodHere()) {
    	RAT_EatFood();
    	correctChoice = choice;
  	} else {
    	correctChoice = notChosen;
  	}

  	// move us (the rat) back to the center of the maze
	RAT_Rotate(180);
	
	if (! RAT_Move()) {
		Debug('e', "RATNET2RatPlugin::Step: Could not move back to center of maze");
	}
	
  	// If there was no error, then one of distA1 and distA2 will be zero
  	// and the choice we made will be the correct choice.  
  	bool noError = ((min_c(distA1, distA2) == 0.0) && (correctChoice == choice));

  	// If we did make an error, then apply the back-propagation to modify the weights
  	// in the network. Otherwise, don't bother modifying the weights.
  	if (! noError) {
    	// run a single iteration of our back prop model in incremental mode
    	CObXPlex* theDesiredOutput = correctChoice;
    	neuralNet.Analyze(theSample, theDesiredOutput, LEARNING_RATE);
    	neuralNet.Train();
  	}

  	for (int j=0; j < NUMBER_INPUT_UNITS; j++) {
    	delete (*theResult)[j++];
  	}

  	for (int j=0; j < NUMBER_INPUT_UNITS; j++) {
    	delete (*theSample)[j++];
  	}

  	for (int j=0; j < NUMBER_OUTPUT_UNITS; j++) {
    	delete (*A1)[j++];
  	}
  
  	for (int j=0; j < NUMBER_OUTPUT_UNITS; j++) {
    	delete (*A2)[j++];
  	}

  	theResult->clear();
  	delete theResult;

  	theSample->clear();
  	delete theSample;
  
  	A1->clear();
  	delete A1;
  
  	A2->clear();
  	delete A2;
}

/*******************************************************
* Main function for the rat thread.
*******************************************************/
long RATNET2RatPlugin::PLUGIN_Start()
{	
	Debug('n', "RATNET2RatPlugin::PLUGIN_Start: In the RATNET2 Rat Thread!!!");
	
	trialNumber = 1; // for evaluation purposes
	
	for (;;) {
		// getting trial type just so we can keep track of evaluation results in database
		trialType = RAT_GetCurrentTrialType();
		Step();
		trialNumber++;	
		//RAT_Wait(0.25); // sleep 1/4 of a second
	}

	return 0;
}

// RATNET2Plugin.cpp ends
